/**************************************************************************/
/* FILE NAME: mpc5500_xcptn.c               COPYRIGHT (c) Freescale 2004  */
/*                                                All Rights Reserved     */
/* DESCRIPTION:                                                           */
/* This file contains function calls for the MPC5500 exceptions and       */
/*  interrupts.                                                           */
/*========================================================================*/
/* REV      AUTHOR       DATE       DESCRIPTION OF CHANGE                 */
/* ---   -----------   ----------   ---------------------                 */
/* 0.1   G. Jackson    12/Jul/04     Initial version                      */ 
/* 1.0   G. Jackson    17/Aug/04     Added xcptn_xmpl function for        */
/*                                     exception and interrupt examples.  */
/* 1.1   C. Baker      19/Jul/06     Removed mpc5554.h #include, added    */
/*                                     software interrupt call            */
/**************************************************************************/

#include "mpc5500_usrccdcfg.h"
#include "mpc5500_xcptn.h"

/**************************************************************************/
/*                  C Code Variables and Definitions                      */
/**************************************************************************/
extern void IVOR4Handler (void);
extern vuint32_t __IV_ADDR;     /* Defined in the linker file */
extern uint32_t IntcIsrVectorTable[];

extern vuint32_t highest_freq;
extern vuint32_t highest_freq1;

extern int inout_buffer[2*1024];

extern vuint16_t RQUEUE0[1024];
extern vuint16_t RQUEUE1[1024];

extern int can_complete;
extern int loop_cnt;

/**************************************************************************/
/*                       C Code Functions                                 */
/**************************************************************************/

/************************************************************************/
/* FUNCTION     : initIrqVectors                                        */
/* PURPOSE      : This function intializes the IVPR and IVOR4 registers */
/*                  for exceptions.                                     */
/* INPUT NOTES  : None                                                  */
/* RETURN NOTES : None                                                  */
/* WARNING      : None                                                  */
/************************************************************************/

void initIrqVectors(void) {
  asm ("lis r5, __IV_ADDR@h");   /* IVPR = address base used with IVOR's */
  asm ("mtIVPR r5 ");
  asm ("lis r5, IVOR4Handler@h"); /* IVOR4 = address of handler */ 							
  asm ("ori r5, r5, IVOR4Handler@l");	
  asm ("mtIVOR4 r5");					
}

/**************************************************************************/
/* FUNCTION     : initINTC                                                */
/* PURPOSE      : This function intializes the INTC for software vector   */
/*                 mode.  Set up INTC vector table base address to start  */
/*                 at the beginning of L2SRAM.                            */
/* INPUT NOTES  : None                                                    */
/* RETURN NOTES : None                                                    */
/* WARNING      : None                                                    */
/**************************************************************************/
void initINTC(void) {	
  INTC.MCR.B.HVEN = 0;	/* Initialize INTC for software vector mode */
  INTC.MCR.B.VTES = 0;	/* Use the default vector table entry offsets of 4 bytes */
  INTC.IACKR.R = (uint32_t) &IntcIsrVectorTable[0]; /* Set INTC ISR vector table base addr. */
}

/************************************************************************/
/* FUNCTION     : enableIrq                                             */
/* PURPOSE      : This function sets INTC's current priority to 0.      */
/*                  External interrupts are enabled.                    */
/* INPUT NOTES  : None                                                  */
/* RETURN NOTES : None                                                  */
/* WARNING      : None                                                  */
/************************************************************************/

void enableIrq(void) {
  
	INTC.PSR[51].R = 0x1;			  	/* Raise eMIOS CH0 IRQ priority = 1 */	
	INTC.PSR[14].R = 0x2;			  	/* Raise eDMA CH3 IRQ priority = 2 */
	INTC.PSR[179].R = 0x3;			  	/* Raise FlexCAN C MB3 IRQ priority = 3 */
	INTC.PSR[156].R = 0x4;			  	/* Raise FlexCAN A MB1 IRQ priority = 4 */
	INTC.CPR.B.PRI = 0;					/* Ensure INTC's current priority is 0 */

  asm(" wrteei 1");		/* Enable external interrupts */
}

/**************************************************************************/
/* FUNCTION     : emios_ch0_ISR                                           */
/* PURPOSE      : ISR 1 - Clears the eMIOS CH0 interrupt and enables 	  */
/*				  eDMA CH1 and 3 to start eQADC transfer				  */
/**************************************************************************/
void emios_ch0_ISR(void) {
	
	// Clear eMIOS CH0 interrupt
	EMIOS.CH[0].CSR.B.FLAG = 0x1;  

	// Enable eDMA to retrieve eQADC RQUEUE0/1
	EDMA.SERQR.R = 0x1;
	EDMA.SERQR.R = 0x3; 

}

/**************************************************************************/
/* FUNCTION     : edma_ch3_ISR											  */
/* PURPOSE      : ISR 2 - Performs the SPE FFT, transmits via DSPI modules*/                            
/*				  and then receives via FlexCAN modules					  */
/**************************************************************************/
void edma_ch3_ISR(void) {
	
	int i;
	
	// Clear eDMA_CH3 interrupt
	EDMA.CIRQR.R = 0x3;	

	// Enable SPE
	asm("mfmsr r3");
	asm("lis r4, 0x0200");
	asm("or r3,r4,r3");
	asm("mtmsr r3");
	
	Check_eQADC_End();					// Wait for scan to complete
			
	//----- Calculate Freq for RQUEUE0 ------
	for(i=0; i<1024; i++) 
	{
		inout_buffer[1024+i] =	(0|(RQUEUE0[i] << 16));
	}
	
	highest_freq = FFT_freq(inout_buffer);
	//---------------------------------------
	
	
	// Clear inout buffer
	for(i=0; i<2048; i++) 
	{
		inout_buffer[i] = 0;
	}
	
	//----- Calculate Freq for RQUEUE1 ------
	for(i=0; i<1024; i++) 
	{
		inout_buffer[1024+i] =	(0|(RQUEUE1[i] << 16));
	}
	
	highest_freq1 = FFT_freq(inout_buffer);
	//---------------------------------------
	
	// Perform DSPI communication
    DSPI_test();
    
	// Perform FlexCAN communication
    FlexCAN_test();	
    
}

/**************************************************************************/
/* FUNCTION     : cana_buf1_ISR											  */
/* PURPOSE      : ISR 4 - This function checks that the data received on  */
/*				  CAN A is	valid										  */                            
/**************************************************************************/
void cana_buf1_ISR(void) {
	
	cana_rec();
 
}

/**************************************************************************/
/* FUNCTION     : canc_buf3_ISR											  */
/* PURPOSE      : ISR 3 - This function checks that the data received on  */
/*				  CAN C is	valid										  */
/**************************************************************************/
void canc_buf3_ISR(void) {
	
	canc_rec();
	 
}

/******************************************************************************/
/* FUNCTION     : xcptn_xmpl                                                  */
/* PURPOSE      : This function serves as an example on setting up exceptions */
/*                  and interrupts in code.                                   */
/* INPUTS NOTES : None                                                        */ 
/* RETURN NOTES : None                                                        */
/* WARNING      : This function is called from main(). It is commented out to */
/*                  bypass the interrupt function.                            */
/******************************************************************************/

void xcptn_xmpl(void) {
    init_LED(PIN_EMIOS10, PIN_HI);   /* Initialize the GPIO ports for output. */
    initIrqVectors();         /* Initialize exceptions: load IVPR only for this example */
    initINTC();               /* Initialize INTC for software vector mode */
    enableIrq();              /* Ensure INTC current prority=0 & enable IRQ */
}

/****************************************************/
/****************************************************/
